/* 
 * Copyright (c) 2016, Charlie Kwon, redisGate.com <redisgate at gmail dot com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name of Redis nor the names of its contributors may be used
 *     to endorse or promote products derived from this software without
 *     specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * 한글 BSD 라이선스
 *
 * 저작권 (c) 2016, 권대욱 <redisgate@gmail.com>, redisGate.com
 * All rights reserved.
 *
 * 소스코드와 바이너리 형태의 재배포와 사용은 수정 여부와 관계없이
 * 다음 조건을 충족할 때 가능하다.
 * 
 *   * 소스코드를 재배포하기 위해서는 반드시 위의 저작권 표시, 지금 보이는 조건들과 
 *     다음과 같은 면책조항을 유지하여야만 한다.
 *   * 바이너리 형태의 재배포는 배포판과 함께 제공되는 문서 또는 다른 형태로 위의 저작권 표시,
 *     지금 보이는 조건들과 다음과 같은 면책조항을 명시해야 한다.
 *   * 사전 서면 승인 없이는 저자의 이름이나 기여자들의 이름을 이 소프트웨어로부터 파생된
 *     제품을 보증하거나 홍보할 목적으로 사용할 수 없다.
 *
 * 본 SW는 저작권자와 기여자들에 의해 "있는 그대로" 제공될 뿐이며, 상품가치나 특정한 목적에
 * 부합하는 묵시적 보증을 포함하여(단, 이에 제한되지 않음), 어떠한 형태의 보증도 하지 않는다.
 * 어떠한 경우에도 재단과 기여자들은 제품이나 서비스의 대체 조달, 또는 데이터, 이윤, 사용상의
 * 손해, 업무의 중단 등을 포함하여(단, 이에 제한되지 않음), 본 소프트웨어를 사용함으로써 발생한
 * 직접적이거나, 간접적 또는 우연적이거나, 특수하거나, 전형적이거나, 결과적인 피해에 대해,
 * 계약에 의한 것이든, 엄격한 책임, 불법행위(또는 과실 및 기타 행위를 포함)에 의한 것이든, 이와
 * 여타 책임 소재에 상관없이, 또한 그러한 손해의 가능이 예견되어 있었다 하더라도 전혀 책임을
 * 지지 않는다.
 *
 */

#include "server.h"
#include "cluster.h"

#include <fcntl.h>
#include <sys/stat.h>


/*-----------------------------------------------------------------------------
 * Manage Users: login, save
 *----------------------------------------------------------------------------*/

/* 문자열를 sort하기 위해서 qsort용 sort founction을 만든다. 
 * ip, userid 비교에 사용된다.
 * 2016.07.23 Charlie Kwon
 */
int strSortCmp(const void *s1, const void *s2) {
  	char const *so1 = (char const *)s1;
	char const *so2 = (char const *)s2;
	
	return strcmp(so1, so2);
}



/* Beginning of Users: 2016.06.21 Charlie Kwon, redisGate.com */
/* loginCommand 
 */
void loginCommand(client *c) {
	users *user;
	sds id, pw;

    if (!server.requirelogin) {
        addReplyError(c,"Client sent LOGIN, but login not enabled. config set requirelogin yes");
		return;
    }

	id = sdsnew(c->argv[1]->ptr);
	pw = sdsnew(c->argv[2]->ptr);

	if (dictSize(server.users)) {
		user = (users*) dictFetchValue(server.users, id);
		if (user) {
			/* check login fail count, redis user는 login fail count를 check하지 않는다.  */
			if (server.login_fail_count > 0 && user->loginFailCnt >= server.login_fail_count
				&& strcmp(c->argv[1]->ptr, "redis") ) {
      			addReplyError(c,"로그인 실패 횟수를 초과하였습니다.");
				return;
			}
			c->user = user;
			c->login_ok = 1;
			/* Compare password: pw is original password, user->pw is hash password */
			if (checkcrypt(pw, user->pw)) {
				c->user->loginCnt++;
				c->user->lastLoginTime = time(NULL);

				if (!sdscmp(id,sdsnew("redis")))
					c->user->admin = 1;
				else
					c->user->admin = 0;

      			addReply(c,shared.ok);
				return;
			}
			c->login_ok = 0;
			c->user->loginFailCnt++;
      		addReplyError(c,"Invalid userid or password");
			return;
		} else {
			c->login_ok = 0;
      		addReplyError(c,"Invalid userid or password");
			return;
		}
	}
	c->login_ok = 0;
   	addReplyError(c,"No user list");
	return;
}
/* End of Users: 2016.07.18 Charlie Kwon, redisGate.com */



/* Beginning of Users: 2016.07.27 Charlie Kwon, redisGate.com */
/* loginxCommand 
 */
void loginxCommand(client *c) {
	users *user;
	sds id, pw;

    if (!server.requirelogin) {
        addReplyError(c,"Client sent LOGINX, but login not enabled. config get/set or redis.conf requirelogin");
		return;
    }

	id = sdsnew(c->argv[1]->ptr);
	pw = sdsnew(c->argv[2]->ptr);

	if (dictSize(server.users)) {
		user = (users*) dictFetchValue(server.users, id);
		if (user) {
			/* check login fail count */
			if (server.login_fail_count > 0 && user->loginFailCnt > server.login_fail_count) {
      			addReplyError(c,"로그인 실패 횟수를 초과하였습니다.");
				return;
			}
			c->user = user;
			c->login_ok = 1;
			/* Compare password: pw is original password, user->pw is hash password */
			if (!sdscmp(pw, user->pw)) {
				c->user->loginCnt++;
				c->user->lastLoginTime = time(NULL);

				if (!sdscmp(id,sdsnew("redis")))
					c->user->admin = 1;
				else
					c->user->admin = 0;

      			addReply(c,shared.ok);
				return;
			}
			c->login_ok = 0;
			c->user->loginFailCnt++;
      		addReplyError(c,"Invalid userid or password");
			return;
		} else {
			c->login_ok = 0;
      		addReplyError(c,"Invalid userid or password");
			return;
		}
	}
	c->login_ok = 0;
   	addReplyError(c,"No user list");
	return;
}
/* End of Users: 2016.07.27 Charlie Kwon, redisGate.com */



/*------------------------------------------------------------------------------
 * Beginning of Users: 2016.07.19 Charlie Kwon, redisGate.com 
 * logoutCommand 
 */
void logoutCommand(client *c) {
	if (c->login_ok) {
		c->login_ok = 0;
		c->user->admin = 0;
		c->user = NULL;
    	addReply(c,shared.ok);
		return;
	}
	addReply(c,shared.nologinerr);

}
/* End of Users: 2016.07.20 Charlie Kwon, redisGate.com */



/* Beginning of Users: 2016.06.21 Charlie Kwon, redisGate.com */
/* chpwCommand: 자신의 password를 변경한다. 
 */
void chpwCommand(client *c) {
	sds chpw = sdsnew(c->argv[1]->ptr);

	/* Check length of password */
	if (sdslen(chpw) < CONFIG_USER_PASSWORD_MIN_LEN || sdslen(chpw) > CONFIG_USER_PASSWORD_MAX_LEN) {
		addReplyErrorFormat(c, "Password length: %d ~ %d", CONFIG_USER_PASSWORD_MIN_LEN, CONFIG_USER_PASSWORD_MAX_LEN); 
		return;
	}

    if (c->login_ok) {
		sdsfree(c->user->pw);
		c->user->pw = sdsnew(bfcrypt(chpw));
      	addReply(c,shared.ok);
		return;
	}
	addReply(c,shared.nologinerr);
}
/* End of Users: 2016.07.18 Charlie Kwon, redisGate.com */


/* Beginning of Users: 2016.07.19 Charlie Kwon, redisGate.com */
/* whoamiCommand: 자신의 userid를 조회한다.
 */
void whoamiCommand(client *c) {
    if (c->login_ok) {
		addReplyBulkCString(c, c->user->id);
		return;
	}
	addReply(c,shared.nologinerr);
}
/* End of Users: 2016.07.19 Charlie Kwon, redisGate.com */





/*------------------------------------------------------------------------------
 * Load users.dat filename.
 * users.dat 파일에서 user id와 password를 읽어온다.
 * 서버 시작 시 Loading한다.
 *----------------------------------------------------------------------------*/
#define USERS_MAX_LINE   1024

int loadUsers() {
    char buf[USERS_MAX_LINE+1];
	sds str;
	int lenId, lenPw;
	int redis_user = 0;
	int load_user_cnt = 0;

    /* Load the User */
    FILE *fp;

    if ((fp = fopen(server.usersfile,"r")) == NULL) {
		/* 파일이 없으면 root(redis) id, password를 redis/redis로 넣는다.  */
		users *superUser = zmalloc(sizeof(users));
		superUser->id = sdsnew("redis");
		superUser->pw = sdsnew(bfcrypt("redis"));	/* hashing */
		superUser->loginCnt = 0;
		superUser->admin = 1;
		superUser->loginFailCnt = 0;
		superUser->lastLoginTime = 0;

		dictAdd(server.users, superUser->id, superUser);

		return C_OK;
	}

    while(fgets(buf,USERS_MAX_LINE+1,fp) != NULL) {
		users *user = zmalloc(sizeof(users));
		/* Skip comments and blank lies */
		if (buf[0] == '#' || buf[0] == '\n' || buf[0] == '\0' ) continue;

		str = sdsnew(buf);
		str = sdstrim(str, " \t\r\n");

		user->id = sdsnew(strtok(str," "));
		user->lastLoginTime = atol(strtok(NULL," "));	/* last login time */
		user->pw = sdsnew(strtok(NULL,"\0"));
		user->loginCnt = 0;
		user->admin = 0;
		user->loginFailCnt = 0;


		/* Length of id and password: 5 ~ 20 characters */
		lenId = sdslen(user->id);
		lenPw = sdslen(user->pw);
		
		/* ID는 알파벳과 숫자로만 구성되어야 한다. */
		if (!stralnum(user->id)) {
	 		serverLog(LL_NOTICE,"ID는 알파벳과 숫자만 가능합니다. %s", user->id);
			zfree(user);
    		fclose(fp);
			return C_ERR;
		}

		if( lenId >= CONFIG_USERID_MIN_LEN &&
			lenId <= CONFIG_USERID_MAX_LEN && 
			lenPw == 60) {
			/* redis가 있는지 확인한다. 없으면 뒤에서 추가한다. */	
			if (!strcmp(user->id, "redis"))
				redis_user = 1;

			dictAdd(server.users, user->id, user);
			load_user_cnt++;
		} else {
	 		serverLog(LL_NOTICE,"Userid 또는 Password의 길이가 맞지 않습니다. %s, %s", 
				user->id, user->pw);
			zfree(user);
    		fclose(fp);
			return C_ERR;
		}
	}
	/* redis user가 없으면 추가한다. */
	if (!redis_user) {
		users *superUser = zmalloc(sizeof(users));
		superUser->id = sdsnew("redis");
		superUser->pw = sdsnew(bfcrypt("redis"));	/* hashing */
		superUser->loginCnt = 0;
		superUser->admin = 1;
		superUser->loginFailCnt = 0;
		superUser->lastLoginTime = 0;

		dictAdd(server.users, superUser->id, superUser);
	}

	if (load_user_cnt)
		serverLog(LL_NOTICE, "Users loaded from %s", server.usersfile);

    fclose(fp);

	return C_OK;
}


/*------------------------------------------------------------------------------
 * Save userid and password, 2016.06.19 Charlie Kwon 
 * 사용자를 users.dat 파일에 저장한다.
 * 저장 항목은 userid, password이다.
 * loginCnt는 저장하지 않는다. 
 * loginCnt는 0부터 다시 시작해서 누적한다.
 *----------------------------------------------------------------------------*/
int saveUsers() {
	dictIterator *di;
	dictEntry *de;
	users *user;
   	FILE *fp;
	char tmpfile[256];	/* temp파일에 user를 저장한 후 rename한다. */


	/* Check server.users가 Null인지 확인한다.
	 * Null 이면 리턴한다. */
	if (!server.users) 
		return C_OK;

	snprintf(tmpfile, 256, "temp-users-%d.dat", (int) getpid());
	
	/* 파일 생성 */
   	if ((fp = fopen(tmpfile,"w")) == NULL) {
		serverLog(LL_WARNING, "Opening the temp file for Users write in saveUsers(): %s", strerror(errno));
		return C_ERR;
	}

	di = dictGetIterator(server.users);

	while ((de = dictNext(di)) != NULL ) {
		user = (users *)dictGetVal(de);
		sds buf = sdsnew(de->key);		/* 1 */
		buf = sdscatsds(buf, sdsnew(" "));
		buf = sdscatsds(buf, sdsfromlonglong(user->lastLoginTime));	/* 2 */
		buf = sdscatsds(buf, sdsnew(" "));
		buf = sdscatsds(buf, user->pw);		/* 3 */
		buf = sdscatsds(buf, sdsnew("\r\n"));

		size_t nwritten;
		nwritten = fwrite((char*)buf, sdslen(buf), 1, fp);

		sdsfree(buf);
		if (nwritten <= 0) goto werr;
	}

	if (fflush(fp) == EOF) goto werr;
	if (fsync(fileno(fp)) == -1) goto werr;
	if (fclose(fp) == EOF) goto werr;

	if (rename(tmpfile, server.usersfile) == -1) {
		serverLog(LL_WARNING,"temp 파일을 users.dat 파일로 바꾸는 중 에러가 발생했습니다. %s", strerror(errno));
		unlink(tmpfile);
		return C_ERR;
	}

	return C_OK;

werr:
	serverLog(LL_WARNING,"User 파일을 디스크에 저장 중 에러가 발생했습니다. %s", strerror(errno));
	fclose(fp);
	unlink(tmpfile);
	return C_ERR;
}


