/* Copyright (c) 2016, Charlie Kwon <redisgate at gmail dot com>, redisGate.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name of Redis nor the names of its contributors may be used
 *     to endorse or promote products derived from this software without
 *     specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * 한글 BSD 라이선스
 *
 * 저작권 (c) 2016, 권대욱 <redisgate@gmail.com>, redisGate.com
 *
 * 소스코드와 바이너리 형태의 재배포와 사용은 수정 여부와 관계없이
 * 다음 조건을 충족할 때 가능하다.
 * 
 *   * 소스코드를 재배포하기 위해서는 반드시 위의 저작권 표시, 지금 보이는 조건들과 
 *     다음과 같은 면책조항을 유지하여야만 한다.
 *   * 바이너리 형태의 재배포는 배포판과 함께 제공되는 문서 또는 다른 형태로 위의 저작권 표시,
 *     지금 보이는 조건들과 다음과 같은 면책조항을 명시해야 한다.
 *   * 사전 서면 승인 없이는 저자의 이름이나 기여자들의 이름을 이 소프트웨어로부터 파생된
 *     제품을 보증하거나 홍보할 목적으로 사용할 수 없다.
 *
 * 본 SW는 저작권자와 기여자들에 의해 "있는 그대로" 제공될 뿐이며, 상품가치나 특정한 목적에
 * 부합하는 묵시적 보증을 포함하여(단, 이에 제한되지 않음), 어떠한 형태의 보증도 하지 않는다.
 * 어떠한 경우에도 재단과 기여자들은 제품이나 서비스의 대체 조달, 또는 데이터, 이윤, 사용상의
 * 손해, 업무의 중단 등을 포함하여(단, 이에 제한되지 않음), 본 소프트웨어를 사용함으로써 발생한
 * 직접적이거나, 간접적 또는 우연적이거나, 특수하거나, 전형적이거나, 결과적인 피해에 대해,
 * 계약에 의한 것이든, 엄격한 책임, 불법행위(또는 과실 및 기타 행위를 포함)에 의한 것이든, 이와
 * 여타 책임 소재에 상관없이, 또한 그러한 손해의 가능이 예견되어 있었다 하더라도 전혀 책임을
 * 지지 않는다.
 *
 */

#include "server.h"
#include "cluster.h"

#include <fcntl.h>
#include <sys/stat.h>
/* isspace(): 2016.06.10 Charlie Kwon, redisGate.com */
#include <ctype.h>

/* inet_pton(): 2016.07.01 Charlie kwon, redisGate.com */
#include <arpa/inet.h>


/*-----------------------------------------------------------------------------
 * Util functions
 *----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * Note: This function returns a pointer to a substring of the original string.
 * If the given string was allocated dynamically, the caller must not overwrite
 * that pointer with the returned value, since the original pointer must be
 * deallocated using the same allocator with which it was allocated.  The return
 * value must NOT be deallocated using free() etc. 
 */
/* 이 function은 사용하지 않지만, 나중에 사용할 수 있어서 그대로 둔다. 
 * sdstrim()을 사용한다. 
 */
char *trim(char *str)
{
  	char *end;

  	/* Trim leading space */
	while(isspace(*str)) str++;

	/* All spaces? return */
	if(*str == 0) return str;

	/* Trim trailing space */
	end = str + strlen(str) - 1;
	while(end > str && isspace(*end)) end--;

	/* Write new null terminator */
	*(end+1) = 0;

	return str;
}

/* iplist를 sort하기 위해서 qsort용 sort founction을 만든다. 
 * 2016.07.22 Charlie Kwon
 */
int ipSortCmp(const void *s1, const void *s2) {
	char const *so1 = (char const *)s1;
    char const *so2 = (char const *)s2;

	return strcmp(so1, so2);
}



/* IP가 유효한 것인지 확인한다. 
 * 이것은 IPv4 용이다. IPv6를 사용하려면 AF_INET6를 사용해야 한다.
 */
 int isValidIp4(const char* ip) {
  	struct sockaddr_in sa;
	return inet_pton(AF_INET, ip, &(sa.sin_addr));
}


/*-----------------------------------------------------------------------------
 * IP4인지 확인한다. 
 * Glob-style를 지원한다.
 * 이 함수는 iplist.txt를 읽어서 server.iplist에 넣기 전에 유효한 IP인지 확인할때 사용한다.
 */

int isValidIp4Globstyle(const char* ip) 
{ 
	if (NULL == ip) return 1; 
	
	const char* pos = ip; 
	unsigned char ch = *pos; 
	unsigned short count = 0; 

	while (ch != '\0') { 
		if (!((ch >= '0' && ch <= '9') || ch == '.' || ch == '*' || ch == '?' 
			|| ch == '[' || ch == '-' || ch == ']')) return 0; 

		if (ch == '.') 
			if (++count > 3) return 0; 

		ch = *++pos; 
	} 

	if (count == 3 && *--pos != '.') return 1; 

	return 0; 
}

/*-----------------------------------------------------------------------------
 * Compare IP 
 * Match return 1, Not match return 0
 */
int ipcmp(list *iplist, sds ip) 
{
	listNode *ln;
	listIter li;

	if (listLength(iplist)) {
		listRewind(iplist, &li);
		while((ln = listNext(&li)) != NULL) {
			/* Match return 1 */
			if(stringmatchlen((char*)ln->value, sdslen(ln->value), (char*)ip, sdslen(ip), 0)) {
				return 1;
			}
		}
	}
	/* Not match return 0 */
	return 0;
}



/*-----------------------------------------------------------------------------
 * Load iplist file
 *----------------------------------------------------------------------------*/

/* Load the permission iplist from the iplist.txt file.
 * iplist.txt 파일에서 허용 ip list를 읽어온다.
 * IP는 *, ?를 사용할 수 있다. 즉, glob-style pattern을 사용한다.
 */
#define IPLIST_MAX_LINE   1024

int loadIplist() {
   	char buf[IPLIST_MAX_LINE+1];
	sds ip;

   	/* File pointer */
   	FILE *fp;

   	if ((fp = fopen(server.iplistfile,"r")) == NULL) {
		return C_OK;
	}

   	while(fgets(buf,IPLIST_MAX_LINE+1,fp) != NULL) {
		/* Skip comments and blank lines */
		if (buf[0] == '#' || buf[0] == '\n' || buf[0] == '\0' ) continue;

		ip = sdsnew(buf);
		ip = sdstrim(ip, " \t\r\n");

		if(isValidIp4(ip)){
			dictAdd(server.ipdict, ip, NULL);
		} else
		if (isValidIp4Globstyle(ip)) {
			listAddNodeTail(server.iplist, ip);
		} else {
	 		serverLog(LL_NOTICE,"Invalid IP in iplist.txt: %s", ip);
   			fclose(fp);
			return C_ERR;
		}
	}
   	fclose(fp);

	return C_OK;
}

/* End of load ip file: 2016.07.02 Charlie Kwon, redisGate.com */

